<?php

namespace App\Http\Controllers;

use App\Repositories\UserRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Route;
use App\Adresse;
use App\Demande;
use App\Intervention;
use App\Message;
use App\Societe;
use App\User;


class UserController extends Controller
{
    protected $userRepository;
    
    protected $nbrPerPage = 8;
    
    public $userType = "utilisateur";
    
    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
        $this->middleware('auth');
        //dd(Route::currentRouteName());
        if (strpos(Route::currentRouteName(),'clients')!== false)
            $this->userType = 'client';
            else if (strpos(Route::currentRouteName(),'prestataires')!== false)
                $this->userType = 'prestataire';
                else if (strpos(Route::currentRouteName(),'assistants')!== false)
                    $this->userType = 'assistant';
                else
                    $this->userType = 'user';
        $this->userRepository->restrictType($this->userType);
    }
    
    public function index(Request $request)
    {
        //
        if (!Auth::user()->isAdmin())
            return abort(403, 'Unauthorized action.');
        $users = $this->userRepository->getPaginate($this->nbrPerPage, $request);
        //$links = $users->render();
        if($request->nom)
            $users = $users->appends(["nom"=>$request->nom]);
        
        $user_type = $this->userType;
        
        //$demandes = Demande::with('messages.user')->orderby('created_at','desc')->get();
        
        return view('users.index', compact('users','user_type'));
    }
    
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }
    
    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
    }
    
    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
        if (!Auth::user()->isAdmin()
            && !Auth::user()->assistant
            && Auth::id()!=$id)
            return abort(403, 'Unauthorized action.');
        $user = $this->userRepository->getById($id);
        $adresses =$user->adresses->all();
        $societe=Societe::first();
        $route = route('users.update', $user->id);
        $cnt = $user->interventions()->whereNotNull('scoring')->count();
        $scoring_calcule = ($cnt>0)
        ?number_format($user->interventions->sum('scoring')/$cnt,2,',',' ').'/5'
            :'0';
            
        $demandes = $user->demandes()->orderby('created_at','desc')->paginate(5);
        
        return view('users.show', compact('route','user','adresses','societe','scoring_calcule','demandes'));
        
    }
    
    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
        if (!Auth::user()->isAdmin()
            && Auth::id()!=$id)
            return abort(403, 'Unauthorized action.');
        $user = $this->userRepository->getById($id);
        $adresses =$user->adresses->all();
        $societe=Societe::first();
        $route = route('users.update', $user->id);
        $cnt = $user->interventions()->whereNotNull('scoring')->count();
        $scoring_calcule = ($cnt>0)
                           ?number_format($user->interventions->sum('scoring')/$cnt,2,',',' ').'/5'
                           :'0';
        
        return view('users.edit', compact('route','user','adresses','societe','scoring_calcule'));
    }
    
    
    
    
    
    public function rules($id)
    {
        return [
            'name' => 'required|max:255',
            'surname' => 'required|max:255',
            'email' => 'required|email|max:191|unique:users,email,' . $id,
            'telephone' => 'required|string|max:15',
            'societe' => 'nullable|string|max:100',
            'siret' => 'nullable|string|max:100'
            //
        ];
    }
    
    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        if (!Auth::user()->isAdmin()
            && Auth::id()!=$id)
            return abort(403, 'Unauthorized action.');
        $user = $this->userRepository->getById($id);
        $request->flash();
        $rules=[];
        $inputs = array_filter(array_merge($request->all()));
         
        if (isset($inputs['password'])){
            $rules['password']= 'required|min:8|confirmed';
        }
        else unset($inputs['password']);
        //dd(array_merge($rules,$this->rules($id)));
        
        //if (!(Auth::user()->isAdmin() && Auth::id()==$id))
        if (Auth::id()==$id && $user->isClient())
            $rules['adresse_pardefaut_id']= 'required|max:36';
        

            
        if (!$this->validate($request, array_merge($rules,$this->rules($id)))){
            return redirect(route('users.edit'))->withInput();
        }
        
        
        if (isset($inputs['password']))
            if ($inputs['password-old']
                && Hash::check($inputs['password-old'], Auth::user()->password)) {
                    $inputs['password'] = Hash::make($inputs['password']);
                    
                } else return redirect(route('users.edit',$id))->withErrors(['password-old'=>'error']);
        
        //$inputs['admin'] = intval(isset($inputs['admin']) && $inputs['admin']=='on');
        if (Auth::user()->isAdmin()){
            $inputs['admin'] = intval(isset($inputs['admin']) && $inputs['admin']=='on');
            $inputs['assistant'] = intval(isset($inputs['assistant']) && $inputs['assistant']=='on');
            $inputs['prestataire'] = intval(isset($inputs['prestataire']) && $inputs['prestataire']=='on');
        }
        if (isset($inputs['approved']))
            $inputs['approved'] =1;
        
            
        $user = $this->userRepository->update($id, $inputs);
        
        //Gestion de l'adresse de facturation par défaut
        //if (!(Auth::user()->isAdmin() && Auth::id()==$id)){
        if (Auth::id()==$id && $user->isClient()){
            $user->adresses()->update(['par_defaut'=>0]);
            Adresse::find($inputs['adresse_pardefaut_id'])->update(['par_defaut'=>1]);
        }
        
        if(isset($inputs['adresses_supprimees']))
        //gestion des suppressions d'adresses
            Adresse::destroy(array_filter($inputs['adresses_supprimees'],'strlen'));
        
        
       
        
        /*
         OU retour vue liste
         */
        /*if (Auth::user()->isAdmin())
            return redirect(route('users.index'))->withOk("L'utilisateur " . $inputs['name'] . " a été modifié.");
        else*/
            return redirect(route('home'))->with('message',"Vos informations ont été modifiées");
            
    }
    
    
    private function suppressionComplete($user){
        /*
         * A securiser
         */
        foreach ($user->demandes as $demande){
            $demande->statuts()->detach();
            foreach ($demande->ligne_demandes as $ligne_demande)
                $ligne_demande->delete();
                foreach ($demande->messages as $message)
                    $message->delete();
                    foreach ($demande->paiements as $paiement)
                        $paiement->delete();
                        if ($demande->intervention)
                            $demande->intervention->delete();
                            $demande->delete();
        }
        
        foreach ($user->adresses as $adresse){
            $adresse->delete();
        }
    }
    
    
    
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
        if (!Auth::user()->isAdmin())
            return abort(403, 'Unauthorized action.');
        $user = $this->userRepository->getById($id);
        $nbDemandes = $user->demandes->count();
        if (!$user->isAdmin() && $nbDemandes==0){
            try{
            Demande::where('user_id',$id)->update(['user_id'=>null]);
            Intervention::where('user_id',$id)->update(['user_id'=>null]);
            Adresse::where('user_id',$id)->update(['user_id'=>null]);
            //Message::where('user_id',$id)->update(['user_id'=>null]); user_id not nullable dans bdd
            //$this->suppressionComplete($user);
            
            
            $this->userRepository->destroy($id);
            }
            catch (\Exception $e) {
                return redirect(route('users.index'))->with(['message' => "impossible de supprimer le client " . $user->email .". Code erreur " .$e->getCode() ."."]);
                
            }
            
            return redirect(route('users.index'))->with(['message' => "Le client " . $user->email ." a été supprimé. Nombre de demandes supprimées : ".$nbDemandes]);
        }
        return redirect(route('users.index'))->with(['message' => "impossible de supprimer le client " . $user->email .". Une demande a déjà été lancée."]);
    }
}
